/* Runge-Kutta-Fehlberg-Simo 78 bundle
 * Implemented by Lluis Alseda 
 * Version 3.1, October 2025
 * CHANGES WITH RESPECT TO VERSION 3.0
 * * Minor improvements in the declaration of β's,
 *   Specially the introduction of beta534 to replace beta53 and beta54.
 */
#include "RKF78.h"
#include <math.h>
#include <values.h>

#define ABS(x)   ((x) < 0.0 ? -(x) : (x))
#define MIN(x,y) ((x) < (y) ? (x) : (y))
#define MAX(a,b) ((a) > (b) ? (a) : (b))

/* alpha[i]'s DICTIONARY: α_1 = 0; α_i = alpha[i-2] for i=2,3,4,...,10; α_{11} = 1; α_{12} = 0; α_{13} = 1 */
static double alpha[9] = { 2.e0/27.e0, 1.e0/9.e0, 1.e0/6.e0, 5.e0/12.e0, 0.5e0, 5.e0/6.e0, 1.e0/6.e0, 2.e0/3.e0, 1.e0/3.e0 };

/* β's DICTIONARY is intertwinned with the declaration and initialization:
 *            β_{2,1} = beta21 */
static double beta21 = 2.e0/27.e0,
/*            β_{3,j} = beta3[j-1] for j = 1,2 */
              beta3[2] = { 1.e0/36.e0, 1.e0/12.e0 },
/*            β_{4,1} = beta41; β_{4,2} = 0; β_{4,3} = beta43 */
              beta41 = 1.e0/24.e0, beta43 = 1.e0/8.e0,
/*            β_{5,1} = beta51; β_{5,2} = 0;  β_{5,3} = -beta534; β_{5,4} = beta534 */
              beta51 = 5.e0/12.e0, beta534 = 25.e0/16.e0,
/*            β_{6,1} = beta61; β_{6,2} = β_{6,3} = 0; β_{6,j} = beta6[j-4] for j = 4,5 */
              beta61 = 5.e-2, beta6[2] = { 0.25e0, 0.2e0 },
/*            β_{7,1} = beta71; β_{7,2} = β_{7,3} = 0; β_{7,j} = beta7[j-1] for j = 4,5,6 */
              beta71 = -25.e0/108.e0, beta7[3] = { 125.e0/108.e0, -65.e0/27.e0, 125.e0/54.e0 },
/*            β_{8,1} = beta81; β_{8,2} = β_{8,3} = β_{8,4} = 0; β_{8,j} = beta8[j-5] for j = 5,6,7 */
              beta81    = 31.e0/300.e0, beta8[3] = { 61.e0/225.e0, -2.e0/9.e0, 13.e0/900.e0 },
/*            β_{9,1} = beta91; β_{9,2} = β_{9,3} = 0; β_{9,j} = beta9[j-4] for j = 4,5,6,7,8 */
              beta91    = 2.e0, beta9[5] = { -53.e0/6.e0, 704.e0/45.e0, -107.e0/9.e0, 67.e0/90.e0, 3.e0 },
/*            β_{10,1} = betax1; β_{10,2} = β_{10,3} = 0; β_{10,j} = betax[j-4] for j = 4,5,6,7,8,9 */
              betax1    = -91.e0/108.e0, betax[6] = { 23.e0/108.e0, -976.e0/135.e0, 311.e0/54.e0, -19.e0/60.e0, 17.e0/6.e0, -1.e0/12.e0 },
/*            β_{11,1} = betaxi1; β_{11,2} = β_{11,3} = 0; β_{11,j} = betaxi[j-4] for j = 4,5,6,7,8,9,10 */
              betaxi1   = 2383.e0/4100.e0, betaxi[7] = { -341.e0/164.e0, 4496.e0/1025.e0, -301.e0/82.e0, 2133.e0/4100.e0, 45.e0/82.e0, 45.e0/164.e0, 18.e0/41.e0 },
/*            β_{12,1} = betaxii1; β_{12,2} = β_{12,3} = β_{12,4} = β_{12,5} = 0; β_{12,j} = betaxii[j-6] for j = 6,7,8,9,10; β_{12,11} = 0 */
              betaxii1  = 3.e0/205.e0, betaxii[5] = { -6.e0/41.e0, -3.e0/205.e0, -3.e0/41.e0, 3.e0/41.e0, 6.e0/41.e0 },
/*            β_{13,1} = betaxiii1; β_{13,2} = β_{13,3} = 0; β_{13,j} = betaxiii[j-4] for j = 4,5,6,7,8,9,10; β_{13,11} = 0; β_{13,12} = 1 */
              betaxiii1 = -1777.e0/4100.e0, betaxiii[7] = { -341.e0/164.e0, 4496.e0/1025.e0, -289.e0/82.e0, 2193.e0/4100.e0, 51.e0/82.e0, 33.e0/164.e0, 12.e0/41.e0 };
/* β's end of declaration and initialization */

/* c78[i]'s DICTIONARY: c_1 = c78_0xi; c_2 = c_3 = c_4 = c_5 = 0; c_i = c78[i-6] for i = 6, 7, 8, 9, 10; c_{11} = c78_0xi; c_{12} = c_{13} = 0.
 *                      c*_1 = c*_2 = c*_3 = c*_4 = c*_5 = 0; c*_i = c78[i-6] for i = 6, 7, 8, 9, 10; c_{11} = 0; c_{12} = c_{13} = c78_0xi.     */
static double c78[5] = { 34.e0/105.e0, 9.e0/35.e0, 9.e0/35.e0, 9.e0/280.e0, 9.e0/280.e0 }, c78_0xi = 41.e0/840.e0;

static double rho = 0.9;

/*******************************************************************************************
 ********************************* one dimensional bundle **********************************
 *******************************************************************************************/
double RKF78_1step (double t, double *x, double h, void *ParmsStruct, void (*ODE)(double, double, double *, void *)) {
       double kappa[13]; // kappa's DICTIONARY: κ_i = kappa[i-1]

/* κ_1 */     ODE(t, *x, kappa, ParmsStruct);
/* κ_2 */     ODE(t + alpha[0] * h, *x + h * beta21 * kappa[0], kappa+1, ParmsStruct);
/* κ_3 */     ODE(t + alpha[1] * h, *x + h * (beta3[0] * kappa[0] + beta3[1] * kappa[1]), kappa+2, ParmsStruct);
/* κ_4 */     ODE(t + alpha[2] * h, *x + h * (beta41 * kappa[0] + beta43 * kappa[2]), kappa+3, ParmsStruct);
/* κ_5 */     ODE(t + alpha[3] * h, *x + h * (beta51 * kappa[0] + beta534 * (kappa[3] - kappa[2])), kappa+4, ParmsStruct);
/* κ_6 */     ODE(t + alpha[4] * h, *x + h * (beta61 * kappa[0] + beta6[0] * kappa[3] + beta6[1] * kappa[4]), kappa+5, ParmsStruct);
/* κ_7 */     ODE(t + alpha[5] * h, *x + h * (beta71 * kappa[0] + beta7[0] * kappa[3] + beta7[1] * kappa[4] + beta7[2] * kappa[5]), kappa+6, ParmsStruct);
/* κ_8 */     ODE(t + alpha[6] * h, *x + h * (beta81 * kappa[0] + beta8[0] * kappa[4] + beta8[1] * kappa[5] + beta8[2] * kappa[6]), kappa+7, ParmsStruct);
/* κ_9 */     ODE(t + alpha[7] * h,
                  *x + h * (beta91 * kappa[0] + beta9[0] * kappa[3] + beta9[1] * kappa[4] + beta9[2] * kappa[5] + beta9[3] * kappa[6] + beta9[4] * kappa[7]),
                  kappa+8, ParmsStruct);
/* κ_10 */    ODE(t + alpha[8] * h,
                  *x + h * (betax1 * kappa[0] + betax[0] * kappa[3] + betax[1] * kappa[4] + betax[2] * kappa[5] +
                                                betax[3] * kappa[6] + betax[4] * kappa[7] + betax[5] * kappa[8]),
                  kappa+9, ParmsStruct);
/* κ_11 */    ODE(t + h,
                  *x + h * (betaxi1 * kappa[0] + betaxi[0] * kappa[3] + betaxi[1] * kappa[4] + betaxi[2] * kappa[5] + betaxi[3] * kappa[6] +
                                                 betaxi[4] * kappa[7] + betaxi[5] * kappa[8] + betaxi[6] * kappa[9]),
                  kappa+10, ParmsStruct);
/* κ_12 */    ODE(t,
                  *x + h * (betaxii1 * kappa[0] + betaxii[0] * kappa[5] + betaxii[1] * kappa[6] + betaxii[2] * kappa[7] +
                                                  betaxii[3] * kappa[8] + betaxii[4] * kappa[9]),
                  kappa+11, ParmsStruct);
/* κ_13 */    ODE(t + h,
                  *x + h * (betaxiii1 * kappa[0] + betaxiii[0] * kappa[3] + betaxiii[1] * kappa[4] + betaxiii[2] * kappa[5] + betaxiii[3] * kappa[6] +
                                                   betaxiii[4] * kappa[7] + betaxiii[5] * kappa[8] + betaxiii[6] * kappa[9] + kappa[11]),
                  kappa+12, ParmsStruct);

       double x7 = c78[0] * kappa[5] + c78[1] * kappa[6] + c78[2] * kappa[7] + c78[3] * kappa[8] + c78[4] * kappa[9],
              x8 = x7 + c78_0xi * (kappa[11] + kappa[12]);
              x7 += c78_0xi * (kappa[0] + kappa[10]);

       if (isnan(x7) || isnan(x8)) return MAXDOUBLE;

       *x += h * x8;

       return (ABS(h) * fabs (x8 - x7));
} /* and this is the end */

/* The procedure RKF78_forward integrates forward the vector field.
 * That is, it computes an approximation of x(t + h) starting at x \approx x(t) WITH h POSITIVE.
 * This function assumes WITHOUT CHECKING IT that 0 < hmin <= h <= hmax.
 * If these assumptions fail to happen the results may be unexpected.
 * See the parameters section in the implementation notes. */
int RKF78_forward (double *t, double *x,
                   double *h, double *err,
                   double hmin, double hmax, double tol,
                   void *ParmsStruct,
                   void (*ODE)(double, double, double *, void *))
{   double kappa[13]; // kappa's DICTIONARY: κ_i = kappa[i-1]

/* κ_1 */       ODE(*t, *x, kappa, ParmsStruct); // Computed outside of the loop coz it does depend on h

    while (1) {
/* κ_2 */       ODE(*t + alpha[0] * *h, *x + *h * beta21 * kappa[0], kappa+1, ParmsStruct);
/* κ_3 */       ODE(*t + alpha[1] * *h, *x + *h * (beta3[0] * kappa[0] + beta3[1] * kappa[1]), kappa+2, ParmsStruct);
/* κ_4 */       ODE(*t + alpha[2] * *h, *x + *h * (beta41 * kappa[0] + beta43 * kappa[2]), kappa+3, ParmsStruct);
/* κ_5 */       ODE(*t + alpha[3] * *h, *x + *h * (beta51 * kappa[0] + beta534 * (kappa[3] - kappa[2])), kappa+4, ParmsStruct);
/* κ_6 */       ODE(*t + alpha[4] * *h, *x + *h * (beta61 * kappa[0] + beta6[0] * kappa[3] + beta6[1] * kappa[4]), kappa+5, ParmsStruct);
/* κ_7 */       ODE(*t + alpha[5] * *h, *x + *h * (beta71 * kappa[0] + beta7[0] * kappa[3] + beta7[1] * kappa[4] + beta7[2] * kappa[5]), kappa+6, ParmsStruct);
/* κ_8 */       ODE(*t + alpha[6] * *h, *x + *h * (beta81 * kappa[0] + beta8[0] * kappa[4] + beta8[1] * kappa[5] + beta8[2] * kappa[6]), kappa+7, ParmsStruct);
/* κ_9 */       ODE(*t + alpha[7] * *h,
                    *x + *h * (beta91 * kappa[0] + beta9[0] * kappa[3] + beta9[1] * kappa[4] + beta9[2] * kappa[5] + beta9[3] * kappa[6] + beta9[4] * kappa[7]),
                    kappa+8, ParmsStruct);
/* κ_10 */      ODE(*t + alpha[8] * *h,
                    *x + *h * (betax1 * kappa[0] + betax[0] * kappa[3] + betax[1] * kappa[4] + betax[2] * kappa[5] +
                                                   betax[3] * kappa[6] + betax[4] * kappa[7] + betax[5] * kappa[8]),
                    kappa+9, ParmsStruct);
/* κ_11 */      ODE(*t + *h,
                    *x + *h * (betaxi1 * kappa[0] + betaxi[0] * kappa[3] + betaxi[1] * kappa[4] + betaxi[2] * kappa[5] + betaxi[3] * kappa[6] +
                                                    betaxi[4] * kappa[7] + betaxi[5] * kappa[8] + betaxi[6] * kappa[9]),
                    kappa+10, ParmsStruct);
/* κ_12 */      ODE(*t,
                    *x + *h * (betaxii1 * kappa[0] + betaxii[0] * kappa[5] + betaxii[1] * kappa[6] + betaxii[2] * kappa[7] +
                                                     betaxii[3] * kappa[8] + betaxii[4] * kappa[9]),
                    kappa+11, ParmsStruct);
/* κ_13 */      ODE(*t + *h,
                    *x + *h * (betaxiii1 * kappa[0] + betaxiii[0] * kappa[3] + betaxiii[1] * kappa[4] + betaxiii[2] * kappa[5] + betaxiii[3] * kappa[6] +
                                                      betaxiii[4] * kappa[7] + betaxiii[5] * kappa[8] + betaxiii[6] * kappa[9] + kappa[11]),
                    kappa+12, ParmsStruct);

         double x7 = c78[0] * kappa[5] + c78[1] * kappa[6] + c78[2] * kappa[7] + c78[3] * kappa[8] + c78[4] * kappa[9],
                x8 = x7 + c78_0xi * (kappa[11] + kappa[12]);
                x7 += c78_0xi * (kappa[0] + kappa[10]);

                if (isnan(x7) || isnan(x8)) return 66;

                *err   = *h * fabs (x8 - x7);
                x8 = *x + *h * x8;
         double tolrel = tol * (1.0 + fabs(x8)/100.0); /* It is an absolute tolerance for small values of the components and relative but "retarded two digts" for large values */

                if (*err < tolrel) { *t += *h; *x = x8;
                    if (*err <= ldexp(tolrel, -8)) { *h *= (rho + rho); *h = MIN(hmax, *h); return 0; }
                    *h *= rho * eighthroot(tolrel / *err); *h = MIN(hmax, MAX(hmin, *h));
                    return 0;
                }

                if (*h <= hmin) { *t += *h; *x = x8; *h = hmin; return 0; }

                *h *= rho * eighthroot(tolrel / *err); *h = MAX(hmin, *h);
    } // End of main while loop; going to recompute the RK78 estimates
    return 0;
} /* and this is the end */

/* The procedure RKF78_backward integrates backwards the vector field.
 * That is, it computes an approximation of x(t - h) starting at x \approx x(t) WITH h POSITIVE.
 * This function assumes WITHOUT CHECKING IT that 0 < hmin <= h <= hmax.
 * If these assumptions fail to happen the results may be unexpected.
 * See the parameters section in the implementation notes. */
int RKF78_backward (double *t, double *x,
                    double *h, double *err,
                    double hmin, double hmax, double tol,
                    void *ParmsStruct,
                    void (*ODE)(double, double, double *, void *))
{   double kappa[13]; // kappa's DICTIONARY: κ_i = kappa[i-1]

/* κ_1 */       ODE(*t, *x, kappa, ParmsStruct); // Computed outside of the loop coz it does depend on h

    while (1) {
/* κ_2 */       ODE(*t - alpha[0] * *h, *x - *h * beta21 * kappa[0], kappa+1, ParmsStruct);
/* κ_3 */       ODE(*t - alpha[1] * *h, *x - *h * (beta3[0] * kappa[0] + beta3[1] * kappa[1]), kappa+2, ParmsStruct);
/* κ_4 */       ODE(*t - alpha[2] * *h, *x - *h * (beta41 * kappa[0] + beta43 * kappa[2]), kappa+3, ParmsStruct);
/* κ_5 */       ODE(*t - alpha[3] * *h, *x - *h * (beta51 * kappa[0] + beta534 * (kappa[3] - kappa[2])), kappa+4, ParmsStruct);
/* κ_6 */       ODE(*t - alpha[4] * *h, *x - *h * (beta61 * kappa[0] + beta6[0] * kappa[3] + beta6[1] * kappa[4]), kappa+5, ParmsStruct);
/* κ_7 */       ODE(*t - alpha[5] * *h, *x - *h * (beta71 * kappa[0] + beta7[0] * kappa[3] + beta7[1] * kappa[4] + beta7[2] * kappa[5]), kappa+6, ParmsStruct);
/* κ_8 */       ODE(*t - alpha[6] * *h, *x - *h * (beta81 * kappa[0] + beta8[0] * kappa[4] + beta8[1] * kappa[5] + beta8[2] * kappa[6]), kappa+7, ParmsStruct);
/* κ_9 */       ODE(*t - alpha[7] * *h,
                    *x - *h * (beta91 * kappa[0] + beta9[0] * kappa[3] + beta9[1] * kappa[4] + beta9[2] * kappa[5] + beta9[3] * kappa[6] + beta9[4] * kappa[7]),
                    kappa+8, ParmsStruct);
/* κ_10 */      ODE(*t - alpha[8] * *h,
                    *x - *h * (betax1 * kappa[0] + betax[0] * kappa[3] + betax[1] * kappa[4] + betax[2] * kappa[5] +
                                                   betax[3] * kappa[6] + betax[4] * kappa[7] + betax[5] * kappa[8]),
                    kappa+9, ParmsStruct);
/* κ_11 */      ODE(*t - *h,
                    *x - *h * (betaxi1 * kappa[0] + betaxi[0] * kappa[3] + betaxi[1] * kappa[4] + betaxi[2] * kappa[5] + betaxi[3] * kappa[6] +
                                                    betaxi[4] * kappa[7] + betaxi[5] * kappa[8] + betaxi[6] * kappa[9]),
                    kappa+10, ParmsStruct);
/* κ_12 */      ODE(*t,
                    *x - *h * (betaxii1 * kappa[0] + betaxii[0] * kappa[5] + betaxii[1] * kappa[6] + betaxii[2] * kappa[7] +
                                                     betaxii[3] * kappa[8] + betaxii[4] * kappa[9]),
                    kappa+11, ParmsStruct);
/* κ_13 */      ODE(*t - *h,
                    *x - *h * (betaxiii1 * kappa[0] + betaxiii[0] * kappa[3] + betaxiii[1] * kappa[4] + betaxiii[2] * kappa[5] + betaxiii[3] * kappa[6] +
                                                      betaxiii[4] * kappa[7] + betaxiii[5] * kappa[8] + betaxiii[6] * kappa[9] + kappa[11]),
                    kappa+12, ParmsStruct);

         double x7 = c78[0] * kappa[5] + c78[1] * kappa[6] + c78[2] * kappa[7] + c78[3] * kappa[8] + c78[4] * kappa[9],
                x8 = x7 + c78_0xi * (kappa[11] + kappa[12]);
                x7 += c78_0xi * (kappa[0] + kappa[10]);

                if (isnan(x7) || isnan(x8)) return 66;

                *err   = *h * fabs (x8 - x7);
                x8 = *x - *h * x8;
         double tolrel = tol * (1.0 + fabs(x8)/100.0);

                if (*err < tolrel) { *t -= *h; *x = x8;
                    if (*err <= ldexp(tolrel, -8)) { *h *= (rho + rho); *h = MIN(hmax, *h); return 0; }
                    *h *= rho * eighthroot(tolrel / *err); *h = MIN(hmax, MAX(hmin, *h));
                    return 0;
                }

                if (*h <= hmin) { *t -= *h; *x = x8; *h = hmin; return 0; }

                *h *= rho * eighthroot(tolrel / *err); *h = MAX(hmin, *h);
    } // End of main while loop; going to recompute the RK78 estimates
    return 0;
} /* and this is the end */

/*******************************************************************************************
 *********************************  n-dimensional bundle  **********************************
 *******************************************************************************************/
double RKF78_VF_1step (double t, double x[], unsigned dim, double h, void *ParmsStruct, void (*VectorField)(double, double *, unsigned, double *, void *)) { register int i;
       double allkappasstorage[13*dim], extrapolatedXk[dim],
              * vectkappa[13] = { allkappasstorage,  // kappa's DICTIONARY: κ_i = vectkappa[i-1]
                                  allkappasstorage + dim, allkappasstorage + dim + dim, allkappasstorage + 3*dim,
                                  allkappasstorage + 4*dim, allkappasstorage + 5*dim, allkappasstorage + 6*dim,
                                  allkappasstorage + 7*dim, allkappasstorage + 8*dim, allkappasstorage + 9*dim,
                                  allkappasstorage + 10*dim, allkappasstorage + 11*dim, allkappasstorage + 12*dim };

/* κ_1 */       VectorField(t, x, dim, vectkappa[0], ParmsStruct); // Computed outside of the loop coz it does depend on h

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + h * beta21 * vectkappa[0][i];
/* κ_2 */       VectorField(t + alpha[0] * h, extrapolatedXk, dim, vectkappa[1], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + h * (beta3[0] * vectkappa[0][i] + beta3[1] * vectkappa[1][i]);
/* κ_3 */       VectorField(t + alpha[1] * h, extrapolatedXk, dim, vectkappa[2], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + h * (beta41 * vectkappa[0][i] + beta43 * vectkappa[2][i]);
/* κ_4 */       VectorField(t + alpha[2] * h, extrapolatedXk, dim, vectkappa[3], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + h * (beta51 * vectkappa[0][i] + beta534 * (vectkappa[3][i] - vectkappa[2][i]));
/* κ_5 */       VectorField(t + alpha[3] * h, extrapolatedXk, dim, vectkappa[4], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + h * (beta61 * vectkappa[0][i] + beta6[0] * vectkappa[3][i] + beta6[1] * vectkappa[4][i]);
/* κ_6 */       VectorField(t + alpha[4] * h, extrapolatedXk, dim, vectkappa[5], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + h * (beta71 * vectkappa[0][i] + beta7[0] * vectkappa[3][i] + beta7[1] * vectkappa[4][i] + beta7[2] * vectkappa[5][i]);
/* κ_7 */       VectorField(t + alpha[5] * h, extrapolatedXk, dim, vectkappa[6], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + h * (beta81 * vectkappa[0][i] + beta8[0] * vectkappa[4][i] + beta8[1] * vectkappa[5][i] + beta8[2] * vectkappa[6][i]);
/* κ_8 */       VectorField(t + alpha[6] * h, extrapolatedXk, dim, vectkappa[7], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + h * (beta91 * vectkappa[0][i] + beta9[0] * vectkappa[3][i] + beta9[1] * vectkappa[4][i] +
                                                                           beta9[2] * vectkappa[5][i] + beta9[3] * vectkappa[6][i] + beta9[4] * vectkappa[7][i]); }
/* κ_9 */       VectorField(t + alpha[7] * h, extrapolatedXk, dim, vectkappa[8], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + h * (betax1 * vectkappa[0][i] + betax[0] * vectkappa[3][i] + betax[1] * vectkappa[4][i] +
                                                                          betax[2] * vectkappa[5][i] + betax[3] * vectkappa[6][i] + betax[4] * vectkappa[7][i] +
                                                                          betax[5] * vectkappa[8][i]); }
/* κ_10 */      VectorField(t + alpha[8] * h, extrapolatedXk, dim, vectkappa[9], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + h * (betaxi1 * vectkappa[0][i] + betaxi[0] * vectkappa[3][i] + betaxi[1] * vectkappa[4][i] +
                                                                          betaxi[2] * vectkappa[5][i] + betaxi[3] * vectkappa[6][i] + betaxi[4] * vectkappa[7][i] +
                                                                          betaxi[5] * vectkappa[8][i] + betaxi[6] * vectkappa[9][i]); }
/* κ_11 */      VectorField(t + h, extrapolatedXk, dim, vectkappa[10], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + h * (betaxii1 * vectkappa[0][i] + betaxii[0] * vectkappa[5][i] + betaxii[1] * vectkappa[6][i] +
                                                                          betaxii[2] * vectkappa[7][i] + betaxii[3] * vectkappa[8][i] + betaxii[4] * vectkappa[9][i]); }
/* κ_12 */      VectorField(t, extrapolatedXk, dim, vectkappa[11], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + h * (betaxiii1 * vectkappa[0][i] + betaxiii[0] * vectkappa[3][i] + betaxiii[1] * vectkappa[4][i] +
                                                                          betaxiii[2] * vectkappa[5][i] + betaxiii[3] * vectkappa[6][i] + betaxiii[4] * vectkappa[7][i] +
                                                                          betaxiii[5] * vectkappa[8][i] + betaxiii[6] * vectkappa[9][i] + vectkappa[11][i]); }
/* κ_13 */      VectorField(t + h, extrapolatedXk, dim, vectkappa[12], ParmsStruct);

         double normkappas = 0.e0, err = 0.e0;
         for (i=0; i < dim; i++) { // Reusing extrapolatedXk[i] as x8prediction[i]
              extrapolatedXk[i] = c78[0] * vectkappa[5][i] + c78[1] * vectkappa[6][i] + c78[2] * vectkappa[7][i] + c78[3] * vectkappa[8][i] +
                                  c78[4] * vectkappa[9][i] + c78_0xi * (vectkappa[11][i] + vectkappa[12][i]);
              normkappas += fabs(extrapolatedXk[i]);
              err += fabs(vectkappa[11][i] + vectkappa[12][i] - vectkappa[0][i] - vectkappa[10][i]);
              x[i] += h * extrapolatedXk[i];
         }

         if (isnan(normkappas) || isnan(err)) return MAXDOUBLE;
         return ((ABS(h) * c78_0xi * err)/dim);
} /* and this is the end */

int RKF78_VF_forward (double *t, double x[], unsigned dim,
                      double *h, double *err,
                      double hmin, double hmax, double tol,
                      void *ParmsStruct,
                      void (*VectorField)(double, double *, unsigned, double *, void *))
{   double allkappasstorage[13*dim],
           * vectkappa[13] = { allkappasstorage,  // kappa's DICTIONARY: κ_i = vectkappa[i-1]
                               allkappasstorage + dim, allkappasstorage + dim + dim, allkappasstorage + 3*dim,
                               allkappasstorage + 4*dim, allkappasstorage + 5*dim, allkappasstorage + 6*dim,
                               allkappasstorage + 7*dim, allkappasstorage + 8*dim, allkappasstorage + 9*dim,
                               allkappasstorage + 10*dim, allkappasstorage + 11*dim, allkappasstorage + 12*dim };

/* κ_1 */       VectorField(*t, x, dim, vectkappa[0], ParmsStruct); // Computed outside of the loop coz it does depend on h

    while (1) { double extrapolatedXk[dim]; register int i;

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + *h * beta21 * vectkappa[0][i];
/* κ_2 */       VectorField(*t + alpha[0] * *h, extrapolatedXk, dim, vectkappa[1], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + *h * (beta3[0] * vectkappa[0][i] + beta3[1] * vectkappa[1][i]);
/* κ_3 */       VectorField(*t + alpha[1] * *h, extrapolatedXk, dim, vectkappa[2], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + *h * (beta41 * vectkappa[0][i] + beta43 * vectkappa[2][i]);
/* κ_4 */       VectorField(*t + alpha[2] * *h, extrapolatedXk, dim, vectkappa[3], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + *h * (beta51 * vectkappa[0][i] + beta534 * (vectkappa[3][i] - vectkappa[2][i]));
/* κ_5 */       VectorField(*t + alpha[3] * *h, extrapolatedXk, dim, vectkappa[4], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + *h * (beta61 * vectkappa[0][i] + beta6[0] * vectkappa[3][i] + beta6[1] * vectkappa[4][i]);
/* κ_6 */       VectorField(*t + alpha[4] * *h, extrapolatedXk, dim, vectkappa[5], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + *h * (beta71 * vectkappa[0][i] + beta7[0] * vectkappa[3][i] + beta7[1] * vectkappa[4][i] + beta7[2] * vectkappa[5][i]);
/* κ_7 */       VectorField(*t + alpha[5] * *h, extrapolatedXk, dim, vectkappa[6], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] + *h * (beta81 * vectkappa[0][i] + beta8[0] * vectkappa[4][i] + beta8[1] * vectkappa[5][i] + beta8[2] * vectkappa[6][i]);
/* κ_8 */       VectorField(*t + alpha[6] * *h, extrapolatedXk, dim, vectkappa[7], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + *h * (beta91 * vectkappa[0][i] + beta9[0] * vectkappa[3][i] + beta9[1] * vectkappa[4][i] +
                                                                           beta9[2] * vectkappa[5][i] + beta9[3] * vectkappa[6][i] + beta9[4] * vectkappa[7][i]); }
/* κ_9 */       VectorField(*t + alpha[7] * *h, extrapolatedXk, dim, vectkappa[8], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + *h * (betax1 * vectkappa[0][i] + betax[0] * vectkappa[3][i] + betax[1] * vectkappa[4][i] +
                                                                           betax[2] * vectkappa[5][i] + betax[3] * vectkappa[6][i] + betax[4] * vectkappa[7][i] +
                                                                           betax[5] * vectkappa[8][i]); }
/* κ_10 */      VectorField(*t + alpha[8] * *h, extrapolatedXk, dim, vectkappa[9], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + *h * (betaxi1 * vectkappa[0][i] + betaxi[0] * vectkappa[3][i] + betaxi[1] * vectkappa[4][i] +
                                                                           betaxi[2] * vectkappa[5][i] + betaxi[3] * vectkappa[6][i] + betaxi[4] * vectkappa[7][i] +
                                                                           betaxi[5] * vectkappa[8][i] + betaxi[6] * vectkappa[9][i]); }
/* κ_11 */      VectorField(*t + *h, extrapolatedXk, dim, vectkappa[10], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + *h * (betaxii1 * vectkappa[0][i] + betaxii[0] * vectkappa[5][i] + betaxii[1] * vectkappa[6][i] +
                                                                           betaxii[2] * vectkappa[7][i] + betaxii[3] * vectkappa[8][i] + betaxii[4] * vectkappa[9][i]); }
/* κ_12 */      VectorField(*t, extrapolatedXk, dim, vectkappa[11], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] + *h * (betaxiii1 * vectkappa[0][i] + betaxiii[0] * vectkappa[3][i] + betaxiii[1] * vectkappa[4][i] +
                                                                           betaxiii[2] * vectkappa[5][i] + betaxiii[3] * vectkappa[6][i] + betaxiii[4] * vectkappa[7][i] +
                                                                           betaxiii[5] * vectkappa[8][i] + betaxiii[6] * vectkappa[9][i] + vectkappa[11][i]); }
/* κ_13 */      VectorField(*t + *h, extrapolatedXk, dim, vectkappa[12], ParmsStruct);

         double normkappas = 0.e0, tolrel = 0.e0; *err = 0;
         for (i=0; i < dim; i++) { // Reusing extrapolatedXk[i] as x8prediction[i]
              extrapolatedXk[i] = c78[0] * vectkappa[5][i] + c78[1] * vectkappa[6][i] + c78[2] * vectkappa[7][i] + c78[3] * vectkappa[8][i] +
                                  c78[4] * vectkappa[9][i] + c78_0xi * (vectkappa[11][i] + vectkappa[12][i]);
              normkappas += fabs(extrapolatedXk[i]);
              *err += fabs(vectkappa[11][i] + vectkappa[12][i] - vectkappa[0][i] - vectkappa[10][i]);
              extrapolatedXk[i] = x[i] + *h * extrapolatedXk[i];
              tolrel += fabs(extrapolatedXk[i]);
         }

         if (isnan(normkappas) || isnan(*err)) return 66;

         *err = (*h * c78_0xi * *err)/dim;
         tolrel = tol * (1.0 + tolrel/100.0);

         if (*err < tolrel) { *t += *h;
             for (i=0; i < dim; x[i] = extrapolatedXk[i], i++);
             if (*err <= ldexp(tolrel, -8)) { *h *= (rho + rho); *h = MIN(hmax, *h); return 0; }
             *h *= rho * eighthroot(tolrel / *err); *h = MIN(hmax, MAX(hmin, *h));
             return 0;
         }

         if (*h <= hmin) { *t += *h; *h = hmin;
             for (i=0; i < dim; x[i] = extrapolatedXk[i], i++);
             return 0;
         }

         *h *= rho * eighthroot(tolrel / *err); *h = MAX(hmin, *h);
    } // End of main while loop; going to recompute the RK78 estimates
    return 0;
} /* and this is the end */

int RKF78_VF_backward (double *t, double x[], unsigned dim,
                       double *h, double *err,
                       double hmin, double hmax, double tol,
                       void *ParmsStruct,
                       void (*VectorField)(double, double *, unsigned, double *, void *))
{   double allkappasstorage[13*dim],
           * vectkappa[13] = { allkappasstorage,  // kappa's DICTIONARY: κ_i = vectkappa[i-1]
                               allkappasstorage + dim, allkappasstorage + dim + dim, allkappasstorage + 3*dim,
                               allkappasstorage + 4*dim, allkappasstorage + 5*dim, allkappasstorage + 6*dim,
                               allkappasstorage + 7*dim, allkappasstorage + 8*dim, allkappasstorage + 9*dim,
                               allkappasstorage + 10*dim, allkappasstorage + 11*dim, allkappasstorage + 12*dim };

/* κ_1 */       VectorField(*t, x, dim, vectkappa[0], ParmsStruct); // Computed outside of the loop coz it does depend on h

    while (1) { double extrapolatedXk[dim]; register int i;

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] - *h * beta21 * vectkappa[0][i];
/* κ_2 */       VectorField(*t - alpha[0] * *h, extrapolatedXk, dim, vectkappa[1], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] - *h * (beta3[0] * vectkappa[0][i] + beta3[1] * vectkappa[1][i]);
/* κ_3 */       VectorField(*t - alpha[1] * *h, extrapolatedXk, dim, vectkappa[2], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] - *h * (beta41 * vectkappa[0][i] + beta43 * vectkappa[2][i]);
/* κ_4 */       VectorField(*t - alpha[2] * *h, extrapolatedXk, dim, vectkappa[3], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] - *h * (beta51 * vectkappa[0][i] + beta534 * (vectkappa[3][i] - vectkappa[2][i]));
/* κ_5 */       VectorField(*t - alpha[3] * *h, extrapolatedXk, dim, vectkappa[4], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] - *h * (beta61 * vectkappa[0][i] + beta6[0] * vectkappa[3][i] + beta6[1] * vectkappa[4][i]);
/* κ_6 */       VectorField(*t - alpha[4] * *h, extrapolatedXk, dim, vectkappa[5], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] - *h * (beta71 * vectkappa[0][i] + beta7[0] * vectkappa[3][i] + beta7[1] * vectkappa[4][i] + beta7[2] * vectkappa[5][i]);
/* κ_7 */       VectorField(*t - alpha[5] * *h, extrapolatedXk, dim, vectkappa[6], ParmsStruct);

                for (i=0; i < dim; i++) extrapolatedXk[i] = x[i] - *h * (beta81 * vectkappa[0][i] + beta8[0] * vectkappa[4][i] + beta8[1] * vectkappa[5][i] + beta8[2] * vectkappa[6][i]);
/* κ_8 */       VectorField(*t - alpha[6] * *h, extrapolatedXk, dim, vectkappa[7], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] - *h * (beta91 * vectkappa[0][i] + beta9[0] * vectkappa[3][i] + beta9[1] * vectkappa[4][i] +
                                                                           beta9[2] * vectkappa[5][i] + beta9[3] * vectkappa[6][i] + beta9[4] * vectkappa[7][i]); }
/* κ_9 */       VectorField(*t - alpha[7] * *h, extrapolatedXk, dim, vectkappa[8], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] - *h * (betax1 * vectkappa[0][i] + betax[0] * vectkappa[3][i] + betax[1] * vectkappa[4][i] +
                                                                           betax[2] * vectkappa[5][i] + betax[3] * vectkappa[6][i] + betax[4] * vectkappa[7][i] +
                                                                           betax[5] * vectkappa[8][i]); }
/* κ_10 */      VectorField(*t - alpha[8] * *h, extrapolatedXk, dim, vectkappa[9], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] - *h * (betaxi1 * vectkappa[0][i] + betaxi[0] * vectkappa[3][i] + betaxi[1] * vectkappa[4][i] +
                                                                           betaxi[2] * vectkappa[5][i] + betaxi[3] * vectkappa[6][i] + betaxi[4] * vectkappa[7][i] +
                                                                           betaxi[5] * vectkappa[8][i] + betaxi[6] * vectkappa[9][i]); }
/* κ_11 */      VectorField(*t - *h, extrapolatedXk, dim, vectkappa[10], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] - *h * (betaxii1 * vectkappa[0][i] + betaxii[0] * vectkappa[5][i] + betaxii[1] * vectkappa[6][i] +
                                                                           betaxii[2] * vectkappa[7][i] + betaxii[3] * vectkappa[8][i] + betaxii[4] * vectkappa[9][i]); }
/* κ_12 */      VectorField(*t, extrapolatedXk, dim, vectkappa[11], ParmsStruct);

                for (i=0; i < dim; i++) { extrapolatedXk[i] = x[i] - *h * (betaxiii1 * vectkappa[0][i] + betaxiii[0] * vectkappa[3][i] + betaxiii[1] * vectkappa[4][i] +
                                                                           betaxiii[2] * vectkappa[5][i] + betaxiii[3] * vectkappa[6][i] + betaxiii[4] * vectkappa[7][i] +
                                                                           betaxiii[5] * vectkappa[8][i] + betaxiii[6] * vectkappa[9][i] + vectkappa[11][i]); }
/* κ_13 */      VectorField(*t - *h, extrapolatedXk, dim, vectkappa[12], ParmsStruct);

         double normkappas = 0.e0, tolrel = 0.e0; *err = 0;
         for (i=0; i < dim; i++) { // Reusing extrapolatedXk[i] as x8prediction[i]
              extrapolatedXk[i] = c78[0] * vectkappa[5][i] + c78[1] * vectkappa[6][i] + c78[2] * vectkappa[7][i] + c78[3] * vectkappa[8][i] +
                                  c78[4] * vectkappa[9][i] + c78_0xi * (vectkappa[11][i] + vectkappa[12][i]);
              normkappas += fabs(extrapolatedXk[i]);
              *err += fabs(vectkappa[11][i] + vectkappa[12][i] - vectkappa[0][i] - vectkappa[10][i]);
              extrapolatedXk[i] = x[i] - *h * extrapolatedXk[i];
              tolrel += fabs(extrapolatedXk[i]);
         }

         if (isnan(normkappas) || isnan(*err)) return 66;

         *err = (*h * c78_0xi * *err)/dim;
         tolrel = tol * (1.0 + tolrel/100.0);

         if (*err < tolrel) { *t -= *h;
             for (i=0; i < dim; x[i] = extrapolatedXk[i], i++);
             if (*err <= ldexp(tolrel, -8)) { *h *= (rho + rho); *h = MIN(hmax, *h); return 0; }
             *h *= rho * eighthroot(tolrel / *err); *h = MIN(hmax, MAX(hmin, *h));
             return 0;
         }

         if (*h <= hmin) { *t -= *h; *h = hmin;
             for (i=0; i < dim; x[i] = extrapolatedXk[i], i++);
             return 0;
         }

         *h *= rho * eighthroot(tolrel / *err); *h = MAX(hmin, *h);
    } // End of main while loop; going to recompute the RK78 estimates
    return 0;
} /* and this is the end */

/* eighthrootofpowersoftwo[i] = 2^(i/8) */
static double eighthrootofpowersoftwo[] = { 1.0,
    1.090507732665257659207010655760707978993,
    1.189207115002721066717499970560475915293,
    1.296839554651009665933754117792451159836,
    1.41421356237309504880168872420969807857,
    1.542210825407940823612291862090734841307,
    1.68179283050742908606225095246642979008,
    1.834008086409342463487083189588288856078 };

static double twoto64 = 1.8446744073709551616e19;

/* scalerootfactor[i] = ((256+i)/256)^(1/8) */
static double scalerootfactor[] = { 1.0,
    1.000487448816538668711013867569244074793, 1.00097324084707034743604501683773495937,
    1.001457388111856589232056240986421378863, 1.001939902497953687584112101455088116314,
    1.002420795761194625568019539246451962849, 1.002900079528134067496301843512425335846,
    1.003377765297957220396743200994905202723, 1.003853864444353371080950508389247131301,
    1.00432838821735488361160406240505651945,  1.00480134774514242164980940407031401131,
    1.005272754035817140437516098405269381273, 1.005742617979140574024397146865066186677,
    1.006210950348242924764680111670836216809, 1.006677761801300444068686811666383312895,
    1.00714306288318257587844713631089644477,  1.007606864027069517329527132872536563458,
    1.00806917555604083454559768708206382991,  1.008530007684635755472311332060034890454,
    1.008989370520385746077368642869627757241, 1.009447274065319961109411459268201301031,
    1.00990372821744414590527678183107918786,  1.010358742772193551449390593879966363477,
    1.010812327423860411007372211728970888127, 1.01126449176699651316542431091966587886,
    1.011715245297791392995420652028605006481, 1.012164597415426650320826293956135592606,
    1.012612557423406891669164987725584034197, 1.013059134530867780451558410398989725184,
    1.013504337853861668198161158350767644044, 1.013948176416621268289734795950937043811,
    1.014390659152801822550132058103191045406, 1.014831794906702200292430800401869400744,
    1.015271592434465358933529736681669326622, 1.015710060405258585099176212193169210408,
    1.016147207402433925224928669356403213492, 1.016583041924669205010046693972076897133,
    1.017017572387090027692617411705594514757, 1.017450807122373131977510040210230128422,
    1.017882754381831481556405659608448689025, 1.018313422336481449503835675042560344248,
    1.018742819078092452407783491856357883342, 1.019170952620219380891098697837976595717,
    1.019597830899218165194107662292016488159, 1.020023461775244806712963811100259094355,
    1.020447853033238198816260671834525625722, 1.020871012383887052888230106458874531313,
    1.021292947464581238364661907275508427049, 1.02171366584034783853189299106603545184,
    1.022133175004772217044390825229099123333, 1.0225514823809043834773381820781865576,
    1.022968595322150939762126018001162280289, 1.023384521113152883049852844198615325319,
    1.023799266970649534406044719874111515872, 1.02421284004432885675423443296185206015,
    1.024625247417664419652302834810707551527, 1.025036496108739262799262655044472041447,
    1.025446593071056904627265194709867105583, 1.025855545194339736929974850560083833398,
    1.026263359305315041210154682846649959611, 1.0266700421684888572925304823928465027,
    1.027075600486907929739062127077274655277, 1.027480040902909952719075307299006694774,
    1.027883369998862329222830646584928005691, 1.028285594297889655860674613788187804615,
    1.028686720264590139957674614768590444409, 1.029086754305741151232436393044085955049,
    1.029485702770994106035579050733739054849, 1.029883571953558877915138627600505390051,
    1.030280368090877924170112438029143320849, 1.030676097365290314046657626928507742045,
    1.03107076590468584032141729800303252527,  1.031464379783149392200446193561616590844,
    1.031856945021595763737704138180219088073, 1.032248467588395068341614345093453964899,
    1.032638953399988926389353935065461347347, 1.033028408321497589504035591415383132773,
    1.033416838167318161667501047662783399774, 1.033804248701714074038894621068837699853,
    1.034190645639395967124398312748197170973, 1.034576034646094130794431549815678587528,
    1.034960421339122649569251296878853017277, 1.035343811287935397590293309253538405005,
    1.035726210014674024760890603141666367564, 1.03610762299470807267436339929008766579,
    1.036488055657167356148121483852802443257, 1.036867513385466743447632069384903924546,
    1.037246001517823465612210464978784024667, 1.037623525347767082683961927765640253165,
    1.038000090124642232090262314809213402065, 1.038375701054104281937379060535378084078,
    1.038750363298608009536712758588930651272, 1.039124081977889423104235761059672701996,
    1.039496862169440842246611277787574845928, 1.039868708908979350572827765398388270225,
    1.040239627190908731546650753900660631355, 1.040609621968774996521486803006004979729,
    1.040978698155715611774117314976582727672, 1.04134686062490252927597381625366069501,
    1.041714114209979123909005397589101824332, 1.042080463705491137846580554125313195161,
    1.042445913867311730877148911065141219783, 1.042810469413060733548473439779633241035,
    1.043174135022518198152070970811839536754, 1.043536915338032340750037418069284157521,
    1.043898814964921965668681704305007171501, 1.044259838471873462144373871808552295848,
    1.044619990391332461105779790600535455213, 1.044979275219890238412284527433507579687,
    1.045337697418664949240001103140516191981, 1.045695261413677776713447592388480658415,
    1.046051971596224076321903396248510930147, 1.046407832323239596133797973594583241378,
    1.046762847917661851329437449215418194584, 1.047117022668786730111152936780906617376,
    1.047470360832620406619796619004728362405, 1.047822866632226635086675383067770406385,
    1.048174544258069498079774564271338825434, 1.048525397868351680361782659598724340283,
    1.048875431589348338564296842028299360099, 1.049224649515736635597001857350879095256,
    1.049573055710921007451922033394731643727, 1.049920654207354228830415282596193778887,
    1.05026744900685434281379324031924037931,  1.050613444080917518616713189712291214363,
    1.050958643371026900305210886426515263961, 1.051303050788957508227859647247001279216,
    1.051646670217077253798495636305552876495, 1.051989505508644127181701981581981765079,
    1.052331560488099616367268869269211011133, 1.05267283895135841507663026505839139667,
    1.053013344666094475922320675172490244832, 1.053353081372023464240310398771444681881,
    1.053692052781181667034190430352950812273, 1.054030262578201410509125973891835152567,
    1.054367714420583038731829557126652940806, 1.054704411938963505030081486520108338211,
    1.055040358737381626841118422166828381108, 1.055375558393540053832102492054604951215,
    1.055710014459063998247466378451917481502, 1.05604373045975677558680713715114867035,
    1.056376709895852202882785984736462859879, 1.056708956242263901030805361763503343149,
    1.057040472948831546820710059864546149058, 1.057371263440564119535037000028605726981,
    1.057701331117880186208068134716201676594, 1.058030679356845268884781300587171565988,
    1.058359311509406336478411436634309526832, 1.058687230903623463099404326303038083792,
    1.059014440843898694016749780976786857571, 1.059340944611202159714711539715618422751,
    1.059666745463295477823525217821825939601, 1.059991846634952482031418813254362657465,
    1.060316251338177316427035115995630967829, 1.060639962762419933075721335173731271255,
    1.0609629840747890299999245854561156189,   1.061285318420262466112825352363351685044,
    1.061606968921895189045093877859146765076, 1.061927938681024711207011994471251758742,
    1.062248230777474168841916775900728639541, 1.062567848269752998251749859402719845355,
    1.062886794195255262811200577929834071933, 1.06320507157045566383328086894475548838,
    1.063522683391103267805939510200491205029, 1.063839632632412981986292094867975608622,
    1.064155922249254809815996001816477703969, 1.064471555176340917108026189090938054464,
    1.064786534328410539451402602092382375264, 1.065100862600412760786082793775605204153,
    1.065414542867687192615068119469515678298, 1.065727577986142582844587257578416771544,
    1.066039970792433382775829914105359046212, 1.066351724104134300312923806379410743535,
    1.066662840719912867001501010748904118537, 1.066973323419700046070111223383185705975,
    1.067283174964858908212739144011322159419, 1.06759239809835140142460466818518610037,
    1.067900995544903240785105282701815302729, 1.068208970011166943671041123490020890558,
    1.068516324185883035479989309288282616401, 1.06882306074003945054771366274492247518,
    1.069129182327029152554660461547259555148, 1.06943469158280599833475546232504212613,
    1.069739591126038868624740410920543914119, 1.070043883558264088924030079970850251598,
    1.070347571464036163273398150032568390883, 1.07065065741107684340557959424006433618,
    1.070953143950422555371979213933616613596, 1.071255033616570205406974061234005161442,
    1.071556328927621386454667943262211397481, 1.071857032385425006452278043725827010675,
    1.072157146475718359139488609248440860448, 1.072456673668266657843978929602041723693,
    1.072755616417001052379809340255631499499, 1.073053977160155148887319025908749100168,
    1.073351758320400052140544764637122505613, 1.073648962304977949550804551361553025336,
    1.073945591505834255802900714792461475823, 1.074241648299748336773282387421557298854,
    1.074537135048462831097367898564016264372, 1.074832054098811587475966879783501684033,
    1.075126407782846235538264741793037296744, 1.075420198417961407811045891433137525214,
    1.075713428307018630080645795104538491584, 1.076006099738468897175446901478077778135,
    1.076298214986473950942482557600947028793, 1.076589776311026276941801295714356289141,
    1.07688078595806783613658795734410278946,  1.077171246159607547615556555137676495448,
    1.077461159133837538146742783027923936901, 1.077750527085248174128453595556079767447,
    1.078039352204741891273700864502801491383, 1.078327636669745837138880976779183690026,
    1.07861538264432334138568913027113640559,  1.07890259227928422844720432295300691851,
    1.079189267712293987054678428913984063746, 1.079475411067981810870741596964400438197,
    1.079761024458047524267429213524146593877, 1.080046109981367407083576965189519763196,
    1.080330669724098931995655612540148359788, 1.08061470575978442793896277999735229186,
    1.080898220149453682822193520711604690617, 1.081181214941725498587714053769494576667,
    1.081463692172908211482304567016087791114, 1.081745653867099190218659227988817455749,
    1.082027102036283324526477637651757618675, 1.082308038680430516413496153920274056121,
    1.082588465787592186281235202496154182795, 1.082868385333996805867526394750367772624,
    1.083147799284144469817978590051986623532, 1.083426709590900517521393642678771417499,
    1.083705118195588216679700172933451446885, 1.083983027028080519921188028801665501538,
    1.084260438006890905606648885007270360931, 1.08453735303926331382141236121199948342,
    1.084813774021261188392165783315186363649, 1.085089702837855635615813845588223842881,
    1.08536514136301271023842745172013770625,  1.085640091459779839075505305415263623274,
    1.085914554980371392520284642709263043301, 1.086188533766253414044646957350408152254,
    1.086462029648227517657229609417974586931, 1.086735044446513963145634582883832390955,
    1.087007579970833918794081924424847877388, 1.087279638020490921134448887752499496187,
    1.087551220384451541157328622315469408126, 1.087822328841425266280497226160113852059,
    1.088092965159943607244958703991477257582, 1.08836313109843843898350812155716033488,
    1.088632828405319584383479013971738254165, 1.088902058819051649743987560163078959209,
    1.089170824068230120608519519275813672038, 1.089439125871656726536093433657333847919,
    1.08970696593841408325844277403758721041,  1.089974345967939620556658800084936154018,
    1.090241267650098804078493812030024120755 };

/* Computes the eighth root of a number x \in (0, 256] by using
 * "Range Reduction" adapted to this problem.
 * Algorithm:
 *  Stage 1: For any double normal number x \in (0, 256] there exist unique
 *     p \in \{256,257,258,...,511\}, z \in [0, 1/p), and expnt
 *     such that
            x = (p/256) * (1 + z) * 2^expnt =
              = (p/256) * (1 + z) * 2^(8*int(expnt/8)) * 2^(expnt%8)
 *     Moreover, -1022 <= expnt <= floor(log_2(256)) = 8
 *     Proof: Let expnt := floor(log_2(x)).
 *        Then log_2(x) = expnt + alpha with alpha \in [0,1). Thus,
 *        x = 2^(log_2(x)) = 2^expnt * 2^(alpha) with 2^(alpha) \in [1,2).
 *        Now, set
 *           p := floor(256*2^(alpha)) \in \{256,257,258,...,511\},  and
 *           z := 256*2^(alpha)/p - 1 (that is, 2^(alpha) = (p/256) *(1 + z))
 *        Clearly,   p <= 256*2^(alpha) < p+1 ==>
 *                   1 <= 256*2^(alpha)/p < 1 + 1/p ==>
 *                   0 <= z < 1/p
 *        p/256 <= 2^(alpha) < (p+1)/256.
 *        Moreover, expnt <= floor(log_2(256)) = 8.
 *        Also, since a double normal number is of the form
                x = 2^e * 1.fraction with e >= -1022
 *        we have -1022 <= e = floor(log_2(2^e * 1.fraction)) = expnt
 *                                                                Q.E.D.
 * Consequently, since 256 = 2^8,
        x^(1/8) = (p/256)^(1/8) * (1 + z)^(1/8) * 2^(int(expnt/8)) * 2^((expnt%8)/8)
 *
 *  Stage 2: Approximate (1+z)^(1/8) by
        1 + z/8  - (7*z^2)/128  + (35*z^3)/1024 - (805*z^4)/32768 + (4991*z^5)/262144 - (64883*z^6)/4194304 =
        1 + z*(1 + z*(z*(35 + z*(z*(4991 - 4055.1875*z)/8 - 805)/32)/8 - 7)/16)/8
 *  because 4194304 = 262144 * 16 = 32768 * 8 * 16 = 1024 * 32 * 8 * 16 = 128 * 8 * 32 * 8 * 16    = 8 * 16 * 8 * 32 * 8 * 16;
 *  and 64883/16 = 4055.1875
 *
 * Test 512e+6 evaluations of pow(x, 0.125): 0m24.847s
 *                            eighthroot(x): 0m0.669s
 */
double eighthroot(double x){
       union { double val;
               struct { unsigned long  int mantisa : 52;
                        unsigned short int    exp  : 11; // From 0 to 2047 biased by 1023
                        unsigned char         sign :  1;
               } bits;
       } doublesplittable; doublesplittable.val = x;

       if (x < 0.0 || doublesplittable.bits.exp > 1038U || !doublesplittable.bits.exp) return (0.0/0.0); // In fact the function computes x^(1/8) for 2^e * 1.fraction = x with e <= 15 (that is, for x < 2^15 * 2 = 2^16
       if (doublesplittable.bits.exp == 1) return 0.0; // x is not a normal number

/* Extracting the exponent of two of x. Since x is normal is of the form
 * x = 2^(expnt) * 1.fraction; and this is the same as computing expnt = floor(log_2(x)) */
       int expnt = ((int) doublesplittable.bits.exp) - 1023;
       doublesplittable.bits.exp = 1023U; // Setting the exponent of two of x to zero so doublesplittable.val is now 2^(alpha) = 1.fraction

       x = ldexp(doublesplittable.val, 8);
       unsigned short p = x;
       x = x/p - 1.0; // z := 256*2^(alpha)/p - 1
       double preroot = (1 + x*(1 + x*(x*(35 + x*(x*(4991 - 4055.1875*x)/8 - 805)/32)/8 - 7)/16)/8) * scalerootfactor[p-256]; // (1 + x)^(1/8) * (p/256)^(1/8)

       if ( expnt < 0 ){ unsigned short expntdividedbyeight = (-expnt)/8;
/* Two Comments: -expnt <= 1022 = 127*8 + 6 ==> 0 <= expntdividedbyeight <= 127
 *                expnt  = -8*expntdividedbyeight - (-8*expntdividedbyeight - expnt)
 *                          where (- expnt) - 8*expntdividedbyeight = (- expnt)%8;
 *                          then, 0 <=  (- expnt) - 8*expntdividedbyeight <= 7  */
        preroot /= eighthrootofpowersoftwo[(-expnt) - 8*expntdividedbyeight];
        if (expntdividedbyeight < 64) return preroot/(1UL << expntdividedbyeight); // recall that the largest power of two fitting in an unsigned long is 2^(63)
            return (preroot/twoto64)/(1UL << (expntdividedbyeight - 64)); // expntdividedbyeight - 64 <= 127-64 = 63
        }
        if ( expnt < 8 ) return preroot*eighthrootofpowersoftwo[expnt];
        return preroot * 2.0 * eighthrootofpowersoftwo[expnt-8]; // 8 <= expnt <= 15
}
