#include <stdio.h>
#include "RKF78.h"
#include <math.h>

typedef struct {
    double k;
    double r;
} ODE_Parameters;

/* Solve the equation
        x'(t) = k* x(t) * ( 1 - x(t)/r )
   and the solution is
       x(t) = (r e^(c_1 r + k t))/(e^(c_1 r + k t) - 1)
       x(t) = (r e^(k t))/(e^(k t) + r - 1)
#### Test the RKF78 function (C version)
####   Solve the ODE:
####      x'(t) = k* x(t) * ( 1 - x(t)/r )
====    0.000000000000000    1.000000000000000 0.2375
   0.001000000000000    1.000237526720519 0 0
   0.002800000000000    1.000665209513834 0 2.22044604925031e-16
   0.006040000000000    1.001435475132600 0 2.22044604925031e-16
   0.011872000000000    1.002823368818801 0 0
   0.022369600000000    1.005326169851229 0 0
   0.041265280000000    1.009846125779064 0 2.22044604925031e-16
   0.075277504000000    1.018030570889378 0 2.22044604925031e-16
   0.136499507200000    1.032920979258102 0 2.22044604925031e-16
   0.236499507200000    1.057686651360931 0 2.22044604925031e-16
   0.336499507200000    1.083012208639814 5.55111512312578e-18 2.22044604925031e-16
   0.436499507200000    1.108908668814838 0 2.22044604925031e-16
   0.536499507200000    1.135387187070523 0 0
   0.636499507200000    1.162459053708046 0 0
   0.736499507200000    1.190135691534837 0 0
   0.836499507200000    1.218428652980510 0 4.44089209850063e-16
   0.936499507200000    1.247349616927914 0 2.22044604925031e-16
   1.000000000000000    1.266045955189318 0 2.22044604925031e-16
*/

double SolEDO(double t, void *Params){
       double r = (((ODE_Parameters *) Params)->r), aa = exp((((ODE_Parameters *) Params)->k)*t);
       return r*aa/(aa+r-1);
}

void Example03(double t, double x, double *f, void *Params){ *f = (((ODE_Parameters *) Params)->k)* x * (1 - x/(((ODE_Parameters *) Params)->r)); }

int main ( ) {
    double hmin = 1.e-8, hmax = 0.1, h = 1.e-3, err = 0.0;
    double t=0.0, x = 1.0;
    double tfin = 1.0;

    ODE_Parameters Example03Params = {0.25, 20.0};

    printf ( "#### Test the RKF78 function (C version)\n" );
    printf ( "####   Solve the ODE:\n" );
    printf ( "####      x'(t) = k* x(t) * ( 1 - x(t)/r )\n" );

    Example03(t, x, &err, &Example03Params);
    printf("==== %20.15lf %20.15lf %.15g\n", t, x, err);

    while (t+h < tfin) {
        RKF78_forward(&t, &x, &h, &err, hmin, hmax, 1.e-15, &Example03Params, Example03);
        printf("%20.15lf %20.15lf %.15g %.15g\n", t, x, err, fabs(x-SolEDO(t, &Example03Params)));
   }
   h = tfin - t;
   err = RKF78_1step(t, &x, h, &Example03Params, Example03);
   printf("%20.15lf %20.15lf %.15g %.15g\n", t+h, x, err, fabs(x-SolEDO(t+h, &Example03Params)));
   return 0;
}
