## -*-Tcl-*-
 # ###################################################################
 # 
 #  FILE: "dictMode.tcl"
 #                                last update: 12/06/2002 23:06:28 
 #  Author: Joachim Kock
 #  E-mail: <kock@math.unice.fr>
 # 
 #  Description: 
 # 
 #  Looking up words in dictionaries 
 #  
 # ###################################################################
 ##

alpha::mode Dict 1.0 dummyDict {} { 
} {
} help {
	Console for looking up words in dictionaries.
} maintainer {
	"Joachim Kock" <kock@math.unice.fr> 
}

proc dummyDict {} {}


# ====================================================================
# ====================================================================
# 
# Dictionary management
# 
# ====================================================================
# ====================================================================

set Dict::dictDir "Rimfrost:dics:oldstyle"
set sep "--------------------------------------------------------------------------------"
set availableDics [list Webster-1913 MobyThesaurusRaw daen dafr enda enfr msc]
set initialisedDics [list ]
set currentDic "enda"
set historyDepth 7
# In addition, the following global variables are maintained for each initialised 
# dictionary:
# 
# Dict_global::$dicName::lastPos        the position in the dictionary file of 
#                                       the current entry
# Dict_global::$dicName::historyList    a list whose entries are {pos word} of 
#                                       previously looked-up words.  (Only words
#                                       looked up from scratch: only the proc
#                                       Dict::lookup writes to this variable.)
# Dict_global::$dicName::histIndex      the current index in the history list
#                                       This is used by the history lookup
#                                       mechanism.

proc Dict::initDic { dicName } {
  global Dict::dictDir
  global availableDics
	global initialisedDics
	global currentDic
	global Dict_global::$dicName::historyList
	global Dict_global::$dicName::histIndex
	global Dict_global::$dicName::lastPos
	
	if { [lsearch -exact $availableDics $dicName] < 0 } {
		message "Unknown dictionary"
		return
	}
	source [file join ${Dict::dictDir} $dicName.norm.tcl]
	set Dict_global::$dicName::lastPos 0
	set Dict_global::$dicName::historyList [list ]
	set Dict_global::$dicName::histIndex 0
	if { [lsearch -exact $initialisedDics $dicName] < 0 } {
		lappend initialisedDics $dicName
	}
	set currentDic $dicName
}

Dict::initDic $currentDic

proc Dict::editDict {} {
	global Dict::dictDir
	global currentDic
	global Dict_global::$currentDic::lastPos
  edit -r -w [file join ${Dict::dictDir} $currentDic]
	goto [set Dict_global::$currentDic::lastPos]
}
bind 'o' <csoz> Dict::editDict


# ====================================================================
# ====================================================================
# 
# Procs to be called from anywhere
# 
# ====================================================================
# ====================================================================

# --------------------------------------
# Lookup the word under the cursor, switches to Console of current dictionary
proc consultDictionary { {key ""} } {
	# What to look up?
	if { $key == "" } {
		# is there an open window?:
		if { ![catch {set pos [getPos]}] } { 
			if { [isSelection] } {
				set key [getSelect]
			} else {
# 				set pos [getPos]
				backwardWord 
				set p0 [getPos]
				forwardWord
				set p1 [getPos]
				set key [getText $p0 $p1]
				goto $pos
			}
		}
	}
	
	# Which dictionary?  Is its window open?
	global currentDic
	if { ![catch {bringToFront "* Console ($currentDic) *"}] } {
		icon -f "* Console ($currentDic) *" -o
	} else {
		global fontSize
		set fS $fontSize
		set fontSize 9
		new -g 515 505 504 242 -n "* Console ($currentDic) *" -m Dict -shell 1
		set fontSize $fS
	} 
	# Now we are in the correct console, so just do
	Dict::lookup $key
}
bind 0x1f <z> consultDictionary
bind 0x1f <z> nextWindow "Dict"


# --------------------------------------
# Puts up listpick dialogue to switch dictionary.
# (It also looks up whatever is under the cursor...)
proc changeDictionary { {dict ""} } {
	global currentDic
	if { $dict == "" } {
		global availableDics
		set dict [listpick $availableDics]
	}
	# See if initialisation is needed:
	global initialisedDics
	if { [lsearch -exact $initialisedDics $dict] < 0 } {
		Dict::initDic $dict
	} else {
		set currentDic $dict
	}
	# Lookup something in any case:
	consultDictionary
}
bind 0x1f <sz>  changeDictionary



# ====================================================================
# ====================================================================
# 
# Procs to be called from Dictionary Consoles
# 
# ====================================================================
# ====================================================================

# --------------------------------------
proc Dict::lookup { key } {
	global sep
	global mode
	global Dict::dictDir
	
	# ------------------
	# Determine dictionary:
	set thisWin [lindex [winNames] 0]
	regexp {\((.*)\)} $thisWin dummy dict
	global Dict_global::$dict::historyList
	global Dict_global::$dict::histIndex	
	global Dict_global::$dict::lastPos
	# ------------------
	# Initialisations
	set key [Dict::$dict::normalForm $key]
	set ordstrom [open [file join ${Dict::dictDir} $dict] r]
	seek $ordstrom 0 start
	set lowerlimit [tell $ordstrom]
	seek $ordstrom 0 end
	set upperlimit [tell $ordstrom]
	
	# ------------------
	# Rough binary search, to narrow the interval:
	while { [expr $upperlimit - $lowerlimit >= 200] } {
		set midte [expr ($upperlimit + $lowerlimit) / 2] 
		seek $ordstrom $midte
		gets $ordstrom linje ; #first chunk is junk
		gets $ordstrom linje
		
		if { [ forstErStorst $key [Dict::$dict::normalForm $linje]] } {
			set lowerlimit $midte
		} else {
			set upperlimit $midte
		}
	}
	
	# ------------------
	# So now the goal is within the narrow interval.
	# (In very unlucky cases the goal may actually be a litte after the 
	# interval, but this doesn't matter because we:
	# Go back a little further and read forward linearly:
	if { $lowerlimit >= 200 } {
		seek $ordstrom [expr $lowerlimit - 200]
		gets $ordstrom linje ; #first chunk is junk
	} else {
		seek $ordstrom 0
	}
	set preSpot [tell $ordstrom] ; # position before the line
	gets $ordstrom linje 
	while { [ forstErStorst $key [Dict::$dict::normalForm $linje]] } {
		set preSpot [tell $ordstrom] ; # position before the line
		if { [gets $ordstrom linje] == -1 } {
			break
		}
	}
	
	# ------------------
	# Display the result.
	if { $mode == "Dict" } {
		set res [Dict::$dict::formatVerbet $linje]
		replaceText [minPos] [maxPos] $key \r $sep \r $res \r
		goto [minPos]
		endLineSelect
	} else {
		message "Usage: ConsultDictionary $key"
	}
	close $ordstrom
	
	# ------------------
	# Update the history list.
	set historyList [set Dict_global::$dict::historyList]
	set entry [list $preSpot $key]
	# If the entry is already in the list, delete it:
	set rep [lsearch -exact $historyList $entry] 
	if { $rep >= 0 } {
		set historyList [lreplace $historyList $rep $rep]
	}
	# Insert the word in the list:	
	set historyList [linsert $historyList 0 $entry]
	# Truncate:
	global historyDepth
	if { [llength $historyList] > $historyDepth } {
		set historyList [lreplace $historyList $historyDepth $historyDepth]
	} 
	# Write to the globals:
	set Dict_global::$dict::historyList $historyList
	set Dict_global::$dict::histIndex 0
	set Dict_global::$dict::lastPos $preSpot
}


# --------------------------------------
#  Caution: argument 'back' is 1 for looking backward in time (=forward in the list)
#                             -1 for looking forward in time (=backward in the list)
proc Dict::historyLookup { back } {
	set thisWin [lindex [winNames] 0]
	regexp {\((.*)\)} $thisWin dummy dict
	
	global Dict_global::$dict::historyList
	set historyList [set Dict_global::$dict::historyList]
	
	global Dict_global::$dict::histIndex
	set histIndex [expr [set Dict_global::$dict::histIndex] + $back]
	if { $histIndex < 0 || $histIndex >= [llength $historyList] } {
		message "History exhausted"
		return
	}
	
	set pos [lindex [lindex $historyList $histIndex] 0]
	set key [lindex [lindex $historyList $histIndex] 1]
	global Dict::dictDir
	set ordstrom [open [file join ${Dict::dictDir} $dict] r]
	seek $ordstrom $pos
	global Dict_global::$dict::lastPos
	set Dict_global::$dict::lastPos $pos
	gets $ordstrom linje
	set res [Dict::$dict::formatVerbet $linje]
	global sep
	replaceText [minPos] [maxPos] $key \r $sep \r $res \r
	goto [minPos]
	endLineSelect
	set Dict_global::$dict::histIndex $histIndex
}
proc Dict::histLookBack {} { Dict::historyLookup 1 }
proc Dict::histLookForth {} { Dict::historyLookup -1 }


# --------------------------------------
proc Dict::neighbourEntry { direction } {
	global sep
	global Dict::dictDir
	set thisWin [lindex [winNames] 0]
	regexp {\((.*)\)} $thisWin dummy dict
	global Dict_global::$dict::lastPos
	
	set ordstrom [open [file join ${Dict::dictDir} $dict] r]
	seek $ordstrom [set Dict_global::$dict::lastPos]
	
	if { $direction == "prev" } {
		backgets $ordstrom linje
		set Dict_global::$dict::lastPos [tell $ordstrom] ; # pos before line
	} elseif { $direction == "next" } {
		gets $ordstrom linje ; # first chunk is old stuff
		set Dict_global::$dict::lastPos [tell $ordstrom] ; # pos before line
		gets $ordstrom linje 
	} else {
		return
	}
	
	set key [Dict::$dict::normalForm $linje]
	set res [Dict::$dict::formatVerbet $linje]
	replaceText [minPos] [maxPos] $key \r $sep \r $res \r
	goto [minPos]
	endLineSelect
	close $ordstrom
}
proc Dict::nextEntry { } { Dict::neighbourEntry "next" }
proc Dict::prevEntry { } { Dict::neighbourEntry "prev" }


# ====================================================================
# ====================================================================
# 
# Navigation procs, key press , key bindings
# 
# ====================================================================
# ====================================================================

proc Dict::DblClick {from to shift option control} {
	if {$shift != "0" || $option != "0" || $control != "0"} {
		changeDictionary
	} else {
		Dict::carriageReturn
	}
}

proc Dict::carriageReturn { } {
	if { [pos::compare [lineStart [getPos]] == [minPos]] } { 
		goto [minPos]
		endOfLine 
		set key [getText [minPos] [getPos]]
	} else {      
		if { [isSelection] } {
			set key [getSelect]
		} else {
			set pos [getPos]
			backwardWord 
			set p0 [getPos]
			forwardWord
			set p1 [getPos]
			set key [getText $p0 $p1]
			goto $pos
		}
	}
	Dict::lookup $key
}

bind 0x34 Dict::carriageReturn "Dict"
Bind '\r' Dict::carriageReturn "Dict"


bind 0x7d <c> Dict::nextEntry  "Dict"
bind 0x7e <c> Dict::prevEntry  "Dict"

bind 0x7b <c> Dict::histLookBack "Dict"
bind 0x21 <c> Dict::histLookBack "Dict"
bind 0x7c <c> Dict::histLookForth "Dict"
bind 0x1e <c> Dict::histLookForth "Dict"

# ------------------- Navigation procs ------------------- #
proc Dict::nextWord { } {
	if { [pos::compare [lineStart [getPos]] == [minPos]] } { 
		forwardChar
	} else {
		hiliteWord
	}
}
bind 0x7c  Dict::nextWord  "Dict"

proc Dict::prevWord { } {
	if { [pos::compare [lineStart [getPos]] == [minPos]] } { 
		backwardChar
	} else {
		backwardWord
		hiliteWord
	}
}
bind 0x7b  Dict::prevWord  "Dict"

proc Dict::downWord { } {
	goto [getPos]   
	if { [pos::compare [lineStart [getPos]] == [minPos]] } { 
		nextLine ; # this is to jump over the separatorline
	}
	nextLine
	if { [regexp {[a-z]+} [getText [getPos] [nextLineStart [getPos]]]] } {
		Dict::nextWord
	} else {
		Dict::prevWord
	}
}
bind 0x7d  Dict::downWord  "Dict"

proc Dict::upWord { } {
	goto [getPos]
	previousLine
	if { [regexp {[a-z]+} [getText [getPos] [nextLineStart [getPos]]]] } {
		Dict::nextWord
	} else {
		Dict::prevWord
	}
}
bind 0x7e  Dict::upWord  "Dict"



# ====================================================================
# ====================================================================
# 
# Auxiliary procs
# 
# ====================================================================
# ====================================================================


# ==========================================================================
# The comparison proc.  Expects normalised input.
# ==========================================================================
proc forstErStorst { forst verbet } {
	if { [string compare $forst $verbet] == 1} {
		return 1
	}
	return 0
}



# ==========================================================================
# The proc backgets works just like gets, but reading backwards.
# It sets the insertion point before what it just read, to be ready to 
# continue reading backwards.
# 
# stream is the name of an open stream
# if the argument rec is given, that variable will receive the line preceeding 
# the current insertion point, and the procedure will return a number which is
# the length of the returned string (the string never contains any newline, 
# neither in the beginning nor at the end of the string).  If the argument rec 
# is not given, the return will be of the text chunk found.  If the insertion 
# point is zero (so there is nothing earlier left to read), an empty string is 
# returned in rec, and the value -1.  The new insertion point will be just 
# before the text chunk.
# ==========================================================================
proc backgets { stream {rec ""} } {
	if {$rec != ""} {
		upvar $rec linje
	}
	
	set theSpot [tell $stream]
	if { $theSpot == 0 } {
		set linje ""
		set size -1
	} else {
		
		# first project: find a line start somewhere strictly before $theSpot:
		set newStart $theSpot
		set newSpot $theSpot
		while { $newSpot >= $theSpot } {
			#go back 100 bytes
			set newStart [expr $newStart - 100]
			if { $newStart > 0 } {
				seek $stream $newStart
				gets $stream dummy ; #first chunk is junk
				set newSpot [tell $stream]
			} else {
				set newSpot 0
			} 
		}
		#so now $newSpot is a linestart position strictly before $theSpot
		
		# second project: see if there is a later one, strictly before $theSpot:
		while { $newSpot < $theSpot } {
			set goodSpot $newSpot
			seek $stream $newSpot
			gets $stream linje
			set newSpot [tell $stream] ; # line start position
		}
		#so now $goodSpot is the last linestart position strictly before $theSpot
		#and $linje contains the text we want (perhaps too much)
		
		seek $stream $goodSpot
		if { $newSpot == $theSpot } {
			set size [expr $theSpot - $goodSpot - 1] ; #this is the size in case the
			#original spot was after newline
		} elseif { $newSpot > $theSpot } {
			set size [expr $theSpot - $goodSpot] ; #this is the real size in case the
			#original spot was not after newline
			set linje [string range $linje 0 [expr $size -1]] 
		}
	}
	
	if {$rec != ""} {
		return $size
	} else {
		return $linje
	}
	
}
